// <copyright file="BoardInputSettingsProvider.cs" company="Harris Hill Products Inc.">
//     Copyright (c) Harris Hill Products Inc. All rights reserved.
// </copyright>

namespace Board.Editor.Input
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    
    using Board.Input;
    
    using UnityEditor;
    
    using UnityEngine;
    using UnityEngine.UIElements;
    
    /// <summary>
    /// Provides the user interface for modifying <see cref="BoardInputSettings"/>.
    /// </summary>
    internal class BoardInputSettingsProvider : SettingsProvider
    {
        [SerializeField] private BoardInputSettings m_Settings;
        [SerializeField] private bool m_SettingsIsNotAnAsset;
        
        [NonSerialized] private List<string> m_AvailableInputSettingsAssets;
        [NonSerialized] private GUIContent[] m_AvailableSettingsAssetsHeaderOptions;
        [NonSerialized] private GUIContent[] m_AvailableSettingsAssetsOptions;
        [NonSerialized] private int m_CurrentSelectedInputSettingsAsset = -1;
        [NonSerialized] private Editor m_SettingsEditor;
        
        private const string kSettingsPath = "Project/Board/Input Settings";
        
        /// <summary>
        /// Creates a new instance of the <see cref="BoardInputSettingsProvider"/> class.
        /// </summary>
        /// <param name="path">Path used to place the SettingsProvider in the tree view of the Settings window.</param>
        /// <param name="scope"><see cref="SettingsScope"/> of the SettingsProvider.</param>
        private BoardInputSettingsProvider(string path, SettingsScope scope)
            : base(path, scope)
        {
            label = "Input Settings";
        }
        
        /// <summary>
        /// Creates a new <see cref="BoardInputSettings"/> asset at the specified path
        /// </summary>
        /// <param name="relativePath">The relative path to save the newly created asset.</param>
        private static void CreateNewSettingsAsset(string relativePath)
        {
            var settings = ScriptableObject.CreateInstance<BoardInputSettings>();
            AssetDatabase.CreateAsset(settings, relativePath);
            EditorGUIUtility.PingObject(settings);
            BoardInput.settings = settings;
        }

        /// <summary>
        /// Creates a new <see cref="BoardInputSettings"/> asset.
        /// </summary>
        private static void CreateNewSettingsAsset()
        {
            // Query for file name.
            var projectName = PlayerSettings.productName;
            var path = EditorUtility.SaveFilePanel("Create Board Input Settings File", "Assets",
                string.Join(string.Empty, projectName.Split(default(string[]), StringSplitOptions.RemoveEmptyEntries)) +
                "BoardInputSettings", "asset");
            if (string.IsNullOrEmpty(path))
            {
                return;
            }

            // Make sure the path is in the Assets/ folder.
            path = path.Replace("\\", "/"); // Make sure we only get '/' separators.
            var dataPath = Application.dataPath + "/";
            if (!path.StartsWith(dataPath, StringComparison.CurrentCultureIgnoreCase))
            {
                Debug.LogError($"Board input settings must be stored in Assets folder of the project (got: '{path}')");
                return;
            }

            // Make sure it ends with .asset.
            var extension = Path.GetExtension(path);
            if (string.Compare(extension, ".asset", StringComparison.InvariantCultureIgnoreCase) != 0)
            {
                path += ".asset";
            }

            // Create settings file.
            var relativePath = "Assets/" + path.Substring(dataPath.Length);
            CreateNewSettingsAsset(relativePath);
        }

        /// <summary>
        /// Grab <see cref="BoardInput.settings"/> and set it up for editing.
        /// </summary>
        private void InitializeWithCurrentSettings()
        {
            // Find the set of available assets in the project.
            m_AvailableInputSettingsAssets = FindInputSettingsInProject().ToList();

            // See which is the active one
            m_Settings = BoardInput.settings;
            var currentSettingsPath = AssetDatabase.GetAssetPath(m_Settings);
            if (string.IsNullOrEmpty(currentSettingsPath))
            {
                if (m_AvailableInputSettingsAssets.Count != 0)
                {
                    m_CurrentSelectedInputSettingsAsset = 0;
                    m_Settings = AssetDatabase.LoadAssetAtPath<BoardInputSettings>(m_AvailableInputSettingsAssets[0]);
                    BoardInput.settings = m_Settings;
                }
            }
            else
            {
                m_CurrentSelectedInputSettingsAsset = m_AvailableInputSettingsAssets.IndexOf(currentSettingsPath);
                if (m_CurrentSelectedInputSettingsAsset == -1)
                {
                    // This is odd and shouldn't happen. Solve by just adding the path to the list.
                    m_AvailableInputSettingsAssets.Add(currentSettingsPath);
                    m_CurrentSelectedInputSettingsAsset = m_AvailableInputSettingsAssets.Count - 1;
                }
            }
            
            // Refresh the list of assets we display in the UI.
            m_AvailableSettingsAssetsOptions = new GUIContent[m_AvailableInputSettingsAssets.Count];
            m_AvailableSettingsAssetsHeaderOptions = new GUIContent[m_AvailableInputSettingsAssets.Count];
            for (var i = 0; i < m_AvailableInputSettingsAssets.Count; ++i)
            {
                var name = m_AvailableInputSettingsAssets[i];
                if (name.StartsWith("Assets/"))
                {
                    name = name.Substring("Assets/".Length);
                }

                if (name.EndsWith(".asset"))
                {
                    name = name.Substring(0, name.Length - ".asset".Length);
                }
                
                m_AvailableSettingsAssetsOptions[i] = new GUIContent(name.Replace("/", "\u29f8"));
                m_AvailableSettingsAssetsHeaderOptions[i] = new GUIContent(name);
            }
            
            if (m_Settings != null && (m_SettingsEditor == null || m_SettingsEditor.target != m_Settings))
            {
                if (m_SettingsEditor != null)
                {
                    UnityEngine.Object.DestroyImmediate(m_SettingsEditor);
                }

                m_SettingsEditor = Editor.CreateEditor(m_Settings);
            }
        }
        
        /// <summary>
        /// Selects the <see cref="BoardInputSettings"/> asset at the specified asset path.
        /// </summary>
        /// <param name="path">The path to a <see cref="BoardInputSettings"/> asset.</param>
        private void SelectSettingsAsset(string path)
        {
            m_CurrentSelectedInputSettingsAsset =
                m_AvailableInputSettingsAssets.IndexOf((string)path);
            m_Settings = AssetDatabase.LoadAssetAtPath<BoardInputSettings>(m_AvailableInputSettingsAssets[m_CurrentSelectedInputSettingsAsset]);
            
            if (m_Settings != null && (m_SettingsEditor == null || m_SettingsEditor.target != m_Settings))
            {
                if (m_SettingsEditor != null)
                {
                    UnityEngine.Object.DestroyImmediate(m_SettingsEditor);
                }

                m_SettingsEditor = Editor.CreateEditor(m_Settings);
            }
        }
        
        /// <summary>
        /// Callback invoked by Board's Input system when the settings change.
        /// </summary>
        private void OnSettingsChanged()
        {
            if (m_Settings == null)
            {
                InitializeWithCurrentSettings();
                Repaint();
            }
        }
        
        /// <summary>
        /// Find all <see cref="BoardInputSettings"/> stored in assets in the current project.
        /// </summary>
        /// <returns>List of GUIDs of all <see cref="BoardInputSettings"/> in project.</returns>
        private static string[] FindInputSettingsInProject()
        {
            var guids = AssetDatabase.FindAssets("t:BoardInputSettings");
            return guids.Select(guid => AssetDatabase.GUIDToAssetPath(guid)).ToArray();
        }

        /// <summary>
        /// Callback invoked by Unity when the user clicks on the Settings in the Settings window.
        /// </summary>
        /// <param name="searchContext">Search context in the search box on the Settings window.</param>
        /// <param name="rootElement">Root of the UIElements tree.</param>
        public override void OnActivate(string searchContext, VisualElement rootElement)
        {
            if (m_SettingsEditor != null)
            {
                UnityEngine.Object.DestroyImmediate(m_SettingsEditor);
            }
            InitializeWithCurrentSettings();
            BoardInput.settingsChanged += OnSettingsChanged;
            base.OnActivate(searchContext, rootElement);
        }
        
        /// <summary>
        /// Callback invoked by Unity when the user clicks on another setting or when the Settings window closes.
        /// </summary>
        public override void OnDeactivate()
        {
            if (m_SettingsEditor != null)
            {
                UnityEngine.Object.DestroyImmediate(m_SettingsEditor);
            }
            m_SettingsEditor = null;
            
            BoardInput.settingsChanged -= OnSettingsChanged;
            base.OnDeactivate();
        }
        
        /// <summary>
        /// Callback invoked by Unity to draw the UI.
        /// </summary>
        /// <param name="searchContext">Search context in the search box on the Settings window.</param>
        public override void OnGUI(string searchContext)
        {
            if (m_AvailableInputSettingsAssets.Count == 0)
            {
                EditorGUILayout.HelpBox(
                    "Settings for the Board input system are stored in an asset. Click the button below to create a settings asset you can edit.",
                    MessageType.Info);
                if (GUILayout.Button("Create settings asset", GUILayout.Height(30)))
                {
                    CreateNewSettingsAsset("Assets/BoardInputSettings.asset");
                    InitializeWithCurrentSettings();
                }

                return;
            }

            EditorGUILayout.Space();
            EditorGUILayout.Separator();
            EditorGUILayout.Space();

            Debug.Assert(m_Settings != null);

            if (EditorGUILayout.DropdownButton(
                    m_CurrentSelectedInputSettingsAsset < 0 || m_CurrentSelectedInputSettingsAsset >=
                    m_AvailableSettingsAssetsOptions.Length
                        ? GUIContent.none
                        : m_AvailableSettingsAssetsHeaderOptions[m_CurrentSelectedInputSettingsAsset], FocusType.Passive))
            {
                var menu = new GenericMenu();
                menu.AddDisabledItem(new GUIContent("Available Settings Assets:"));
                menu.AddSeparator("");
                for (var i = 0; i < m_AvailableSettingsAssetsOptions.Length; i++)
                    menu.AddItem(new GUIContent(m_AvailableSettingsAssetsOptions[i]),
                        m_CurrentSelectedInputSettingsAsset == i,
                        (path) =>
                        {
                            SelectSettingsAsset((string)path);
                        }, m_AvailableInputSettingsAssets[i]);
                menu.AddSeparator("");
                menu.AddItem(new GUIContent("New Settings Asset…"), false, CreateNewSettingsAsset);
                menu.ShowAsContext();
                Event.current.Use();
            }

            EditorGUILayout.BeginVertical(EditorStyles.helpBox);
            EditorGUI.indentLevel = 1;
            if (m_SettingsEditor != null)
            {
                m_SettingsEditor.OnInspectorGUI();
            }
            EditorGUI.indentLevel = 0;

            EditorGUILayout.Space();
            EditorGUILayout.EndVertical();
        }
        
        /// <summary>
        /// Opens Unity's project settings window to this provider.
        /// </summary>
        public static void Open()
        {
            SettingsService.OpenProjectSettings(kSettingsPath);
        }
        
        /// <summary>
        /// Creates the settings provider for <see cref="BoardInputSettings"/>.
        /// </summary>
        [SettingsProvider]
        public static SettingsProvider CreateInputSettingsProvider()
        {
            return new BoardInputSettingsProvider(kSettingsPath, SettingsScope.Project);
        }
    }
}
