// <copyright file="BoardInputSettingsBuildProvider.cs" company="Harris Hill Products Inc.">
//     Copyright (c) Harris Hill Products Inc. All rights reserved.
// </copyright>

#if UNITY_ANDROID
namespace Board.Editor.Input
{
    using System;
    using System.IO;
    using System.Linq;

    using Board.Input;

    using UnityEngine;
    using UnityEditor;
    using UnityEditor.Build;
    using UnityEditor.Build.Reporting;

    /// <summary>
    /// Provides a mechanism to configure and validate <see cref="BoardInputSettings"/> on build.
    /// </summary>
    internal class BoardInputSettingsBuildProvider : IPreprocessBuildWithReport, IPostprocessBuildWithReport
    {
        /// <summary>
        /// Gets the relative callback order of the build provider.
        /// </summary>
        public int callbackOrder => 1;

        /// <summary>
        /// Callback invoked by Unity before a Player build is started.
        /// </summary>
        /// <param name="report">A <see cref="BuildReport"/> containing information about the build, such as its target platform and output path.</param>
        public void OnPreprocessBuild(BuildReport report)
        {
            if (BoardInput.settings == null)
            {
                throw new BuildFailedException(
                    "[Board SDK] BoardInputSettings asset not found.\n\n" +
                    "Create one via: Assets > Create > Board > Input Settings\n" +
                    "Then configure it in: Project Settings > Board > Input Settings");
            }

            if (!EditorUtility.IsPersistent(BoardInput.settings))
            {
                throw new BuildFailedException(
                    "[Board SDK] No BoardInputSettings asset is active.\n\n" +
                    "To fix:\n" +
                    "1. Go to Project Settings > Board > Input Settings\n" +
                    "2. Select your BoardInputSettings asset from the dropdown, or click \"Make Active\"\n" +
                    "3. Set the Glyph Model Filename to your .tflite model file");
            }

            var glyphModelFilename = BoardInput.settings.glyphModelFilename;
            if (string.IsNullOrEmpty(glyphModelFilename))
            {
                throw new BuildFailedException(
                    "[Board SDK] No glyph model file configured in BoardInputSettings.\n\n" +
                    "1. Go to Project Settings > Board > Input Settings\n" +
                    "2. Set the 'Glyph Model Filename' to your .tflite model file\n" +
                    "3. Ensure the model file exists in Assets/StreamingAssets/");
            }

            var directory = Path.Combine(Application.dataPath, "StreamingAssets");
            var file = string.Empty;
            try
            {
                file = Directory.GetFiles(directory, glyphModelFilename, SearchOption.AllDirectories)
                    .FirstOrDefault();
            }
            catch (Exception)
            {
                throw new BuildFailedException(
                    $"[Board SDK] StreamingAssets folder not found or inaccessible.\n\n" +
                    $"The glyph model file \"{glyphModelFilename}\" must be placed in Assets/StreamingAssets/.\n\n" +
                    "1. Create the folder: Assets/StreamingAssets/\n" +
                    $"2. Copy your glyph model file ({glyphModelFilename}) into that folder");
            }

            if (string.IsNullOrEmpty(file))
            {
                throw new BuildFailedException(
                    $"[Board SDK] Glyph model file \"{glyphModelFilename}\" not found in StreamingAssets.\n\n" +
                    $"1. Ensure the file exists at: Assets/StreamingAssets/{glyphModelFilename}\n" +
                    "2. Or update the filename in: Project Settings > Board > Input Settings");
            }

            var preloadedAssets = PlayerSettings.GetPreloadedAssets();
            if (!preloadedAssets.Contains(BoardInput.settings))
            {
                var preloadedAssetsList = preloadedAssets.ToList();
                preloadedAssetsList.Add(BoardInput.settings);
                PlayerSettings.SetPreloadedAssets(preloadedAssetsList.ToArray());
            }
        }

        /// <summary>
        /// Callback invoked by Unity after a Player build is completed.
        /// </summary>
        /// <param name="report">A <see cref="BuildReport"/> containing information about the build, such as its target platform and output path.</param>
        public void OnPostprocessBuild(BuildReport report)
        {
            var preloadedAssets = PlayerSettings.GetPreloadedAssets();

            if (preloadedAssets == null || preloadedAssets.Length == 0)
            {
                return;
            }

            var preloadedAssetsList = preloadedAssets.ToList();
            for (var i = 0; i < preloadedAssetsList.Count; i++)
            {
                if (preloadedAssets[i] is BoardInputSettings)
                {
                    preloadedAssetsList.RemoveAt(i);
                    i--;
                }
            }

            PlayerSettings.SetPreloadedAssets(preloadedAssetsList.ToArray());
        }
    }
}
#endif //UNITY_ANDROID